<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Models\RegisterMember;
use Illuminate\Http\Request;

class RegisterMemberController extends Controller
{

    public function blackBelt()
    {
        $members = RegisterMember::where('role', 'Black belt degree')->get();
        return view('admin.register_member.liste_demande', ['members' => $members, 'title' => 'Black Belt Degree']);
    }

    public function selfDefense()
    {
        $members = RegisterMember::where('role', 'Coach self defense')->get();
        return view('admin.register_member.liste_demande', ['members' => $members, 'title' => 'Coach Self Defense']);
    }

    public function bodyGuard()
    {
        $members = RegisterMember::where('role', 'Coach body guard')->get();
        return view('admin.register_member.liste_demande', ['members' => $members, 'title' => 'Coach Body Guard']);
    }
    public function updateValidate(Request $request, $id)
    {
        try {
            $member = RegisterMember::findOrFail($id);
            if($member->validate){
                RegisterMember::where('id',$id)
                    ->update([
                        'validate'=>false
                ]);
            }
            else{
                RegisterMember::where('id',$id)
                    ->update([
                        'validate'=>true
                    ]);
            }
            return redirect()->back()->with('success', 'Registration successfully changed 🎉');

        } catch (\Exception $e) {
            \Log::error('Error updating member validation: ' . $e->getMessage());
            return response()->json(['success' => false, 'error' => $e->getMessage()], 500);
        }
    }
    public function destroy($id)
    {
        $members = RegisterMember::where('id', $id)->delete();
        return redirect()->back()->with('success', 'Registration successfully deleted 🎉');
    }
    public function edit($id)
    {
        $member = RegisterMember::findOrFail($id);
        return view('admin.register_member.edit', compact('member'));
    }

    public function update(Request $request, $id)
    {
        // ✅ 1. Validation des données
        $request->validate([
            'nom' => 'required|string|max:255',
            'prenom' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'tel' => 'nullable|string|max:20',
            'adresse' => 'nullable|string|max:255',
            'nationalite' => 'nullable|string|max:100',
            'role' => 'required|string',
            'photo1' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:5120', // 5MB max
            'certificat' => 'nullable|mimes:pdf,jpg,jpeg,png|max:10240', // 10MB max
        ]);

        // ✅ 2. Récupérer le membre à modifier
        $member = RegisterMember::findOrFail($id);

        // ✅ 3. Préparer les données à mettre à jour
        $dataToUpdate = [
            'nom' => $request->nom,
            'prenom' => $request->prenom,
            'email' => $request->email,
            'tel' => $request->tel,
            'adresse' => $request->adresse,
            'nationalite' => $request->nationalite,
            'role' => $request->role,
        ];

        // ✅ 4. Gérer la photo1 (si nouvelle photo uploadée)
        if ($request->hasFile('photo1')) {
            try {
                $photoData = file_get_contents($request->file('photo1')->getRealPath());
                $dataToUpdate['photo1'] = $photoData;
            } catch (\Exception $e) {
                return redirect()->back()
                    ->with('error', 'Erreur lors du téléchargement de la photo')
                    ->withInput();
            }
        }

        // ✅ 5. Gérer le certificat (si nouveau certificat uploadé)
        if ($request->hasFile('certificat')) {
            try {
                // Supprimer l'ancien certificat s'il existe
                if ($member->certificat && \Storage::disk('public')->exists($member->certificat)) {
                    \Storage::disk('public')->delete($member->certificat);
                }

                // Stocker le nouveau certificat
                $role = str_replace(' ', '_', strtolower($request->role ?? 'autre'));
                $path = $request->file('certificat')->storeAs(
                    "register_members/{$role}",
                    time() . '_certificat.' . $request->file('certificat')->getClientOriginalExtension(),
                    'public'
                );

                $dataToUpdate['certificat'] = $path;
            } catch (\Exception $e) {
                return redirect()->back()
                    ->with('error', 'Erreur lors du téléchargement du certificat')
                    ->withInput();
            }
        }

        // ✅ 6. Mettre à jour le country_code si la nationalité a changé
        if ($request->nationalite !== $member->nationalite) {
            $countryCode = $this->getCountryCodeFromNationality($request->nationalite);
            $dataToUpdate['country_code'] = $countryCode;
        }

        // ✅ 7. Mettre à jour les autres champs optionnels s'ils existent
        if ($request->has('matricule')) {
            $dataToUpdate['matricule'] = $request->matricule;
        }
        if ($request->has('degree')) {
            $dataToUpdate['degree'] = $request->degree;
        }
        if ($request->has('level')) {
            $dataToUpdate['level'] = $request->level;
        }

        // ✅ 8. Effectuer la mise à jour
        try {
            $member->update($dataToUpdate);

            return redirect()->back()->with('success', 'Membre modifié avec succès ! 🎉');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Erreur lors de la modification du membre')
                ->withInput();
        }
    }

// ✅ Méthode helper inchangée
    private function getCountryCodeFromNationality($nationality)
    {
        $countries = [
            'Afghanistan' => 'af',
            'Albania' => 'al',
            'Algeria' => 'dz',
            'Andorra' => 'ad',
            'Angola' => 'ao',
            'Antigua and Barbuda' => 'ag',
            'Argentina' => 'ar',
            'Armenia' => 'am',
            'Australia' => 'au',
            'Austria' => 'at',
            'Azerbaijan' => 'az',
            'Bahamas' => 'bs',
            'Bahrain' => 'bh',
            'Bangladesh' => 'bd',
            'Barbados' => 'bb',
            'Belarus' => 'by',
            'Belgium' => 'be',
            'Belize' => 'bz',
            'Benin' => 'bj',
            'Bhutan' => 'bt',
            'Bolivia' => 'bo',
            'Bosnia and Herzegovina' => 'ba',
            'Botswana' => 'bw',
            'Brazil' => 'br',
            'Brunei' => 'bn',
            'Bulgaria' => 'bg',
            'Burkina Faso' => 'bf',
            'Burundi' => 'bi',
            'Cabo Verde' => 'cv',
            'Cambodia' => 'kh',
            'Cameroon' => 'cm',
            'Canada' => 'ca',
            'Central African Republic' => 'cf',
            'Chad' => 'td',
            'Chile' => 'cl',
            'China' => 'cn',
            'Colombia' => 'co',
            'Comoros' => 'km',
            'Congo (Brazzaville)' => 'cg',
            'Congo (Kinshasa)' => 'cd',
            'Costa Rica' => 'cr',
            'Croatia' => 'hr',
            'Cuba' => 'cu',
            'Cyprus' => 'cy',
            'Czech Republic' => 'cz',
            'Denmark' => 'dk',
            'Djibouti' => 'dj',
            'Dominica' => 'dm',
            'Dominican Republic' => 'do',
            'Ecuador' => 'ec',
            'Egypt' => 'eg',
            'El Salvador' => 'sv',
            'Equatorial Guinea' => 'gq',
            'Eritrea' => 'er',
            'Estonia' => 'ee',
            'Eswatini' => 'sz',
            'Ethiopia' => 'et',
            'Fiji' => 'fj',
            'Finland' => 'fi',
            'France' => 'fr',
            'Gabon' => 'ga',
            'Gambia' => 'gm',
            'Georgia' => 'ge',
            'Germany' => 'de',
            'Ghana' => 'gh',
            'Greece' => 'gr',
            'Grenada' => 'gd',
            'Guatemala' => 'gt',
            'Guinea' => 'gn',
            'Guinea-Bissau' => 'gw',
            'Guyana' => 'gy',
            'Haiti' => 'ht',
            'Honduras' => 'hn',
            'Hungary' => 'hu',
            'Iceland' => 'is',
            'India' => 'in',
            'Indonesia' => 'id',
            'Iran' => 'ir',
            'Iraq' => 'iq',
            'Ireland' => 'ie',
            'Italy' => 'it',
            'Jamaica' => 'jm',
            'Japan' => 'jp',
            'Jordan' => 'jo',
            'Kazakhstan' => 'kz',
            'Kenya' => 'ke',
            'Kiribati' => 'ki',
            'Kuwait' => 'kw',
            'Kyrgyzstan' => 'kg',
            'Laos' => 'la',
            'Latvia' => 'lv',
            'Lebanon' => 'lb',
            'Lesotho' => 'ls',
            'Liberia' => 'lr',
            'Libya' => 'ly',
            'Liechtenstein' => 'li',
            'Lithuania' => 'lt',
            'Luxembourg' => 'lu',
            'Madagascar' => 'mg',
            'Malawi' => 'mw',
            'Malaysia' => 'my',
            'Maldives' => 'mv',
            'Mali' => 'ml',
            'Malta' => 'mt',
            'Marshall Islands' => 'mh',
            'Mauritania' => 'mr',
            'Mauritius' => 'mu',
            'Mexico' => 'mx',
            'Micronesia' => 'fm',
            'Moldova' => 'md',
            'Monaco' => 'mc',
            'Mongolia' => 'mn',
            'Montenegro' => 'me',
            'Morocco' => 'ma',
            'Mozambique' => 'mz',
            'Myanmar (Burma)' => 'mm',
            'Namibia' => 'na',
            'Nauru' => 'nr',
            'Nepal' => 'np',
            'Netherlands' => 'nl',
            'New Zealand' => 'nz',
            'Nicaragua' => 'ni',
            'Niger' => 'ne',
            'Nigeria' => 'ng',
            'North Korea' => 'kp',
            'North Macedonia' => 'mk',
            'Norway' => 'no',
            'Oman' => 'om',
            'Pakistan' => 'pk',
            'Palau' => 'pw',
            'Palestine' => 'ps',
            'Panama' => 'pa',
            'Papua New Guinea' => 'pg',
            'Paraguay' => 'py',
            'Peru' => 'pe',
            'Philippines' => 'ph',
            'Poland' => 'pl',
            'Portugal' => 'pt',
            'Qatar' => 'qa',
            'Romania' => 'ro',
            'Russia' => 'ru',
            'Rwanda' => 'rw',
            'Saint Kitts and Nevis' => 'kn',
            'Saint Lucia' => 'lc',
            'Saint Vincent and the Grenadines' => 'vc',
            'Samoa' => 'ws',
            'San Marino' => 'sm',
            'Sao Tome and Principe' => 'st',
            'Saudi Arabia' => 'sa',
            'Senegal' => 'sn',
            'Serbia' => 'rs',
            'Seychelles' => 'sc',
            'Sierra Leone' => 'sl',
            'Singapore' => 'sg',
            'Slovakia' => 'sk',
            'Slovenia' => 'si',
            'Solomon Islands' => 'sb',
            'Somalia' => 'so',
            'South Africa' => 'za',
            'South Korea' => 'kr',
            'South Sudan' => 'ss',
            'Spain' => 'es',
            'Sri Lanka' => 'lk',
            'Sudan' => 'sd',
            'Suriname' => 'sr',
            'Sweden' => 'se',
            'Switzerland' => 'ch',
            'Syria' => 'sy',
            'Taiwan' => 'tw',
            'Tajikistan' => 'tj',
            'Tanzania' => 'tz',
            'Thailand' => 'th',
            'Timor-Leste' => 'tl',
            'Togo' => 'tg',
            'Tonga' => 'to',
            'Trinidad and Tobago' => 'tt',
            'Tunisia' => 'tn',
            'Turkey' => 'tr',
            'Turkmenistan' => 'tm',
            'Tuvalu' => 'tv',
            'Uganda' => 'ug',
            'Ukraine' => 'ua',
            'United Arab Emirates' => 'ae',
            'United Kingdom' => 'gb',
            'United States' => 'us',
            'Uruguay' => 'uy',
            'Uzbekistan' => 'uz',
            'Vanuatu' => 'vu',
            'Vatican City' => 'va',
            'Venezuela' => 've',
            'Vietnam' => 'vn',
            'Yemen' => 'ye',
            'Zambia' => 'zm',
            'Zimbabwe' => 'zw',
        ];

        // Chercher par texte exact
        if (isset($countries[$nationality])) {
            return $countries[$nationality];
        }

        // Chercher si la nationalité contient un emoji drapeau (🇹🇳 Tunisia)
        preg_match('/^[\x{1F1E6}-\x{1F1FF}]{2} (.+)$/u', $nationality, $matches);
        if (isset($matches[1]) && isset($countries[$matches[1]])) {
            return $countries[$matches[1]];
        }

        return null;
    }

}
