<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Models\Gym;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class RegisterGymController extends Controller
{
    public function index()
    {
        $gyms = Gym::all();
        return view('admin.register_gym.liste_gyms', ['gyms' => $gyms, 'title' => 'Liste des Gyms']);
    }

    public function create()
    {
        return view('admin.register_gym.create_gym', ['title' => 'Ajouter un Gym']);
    }

    public function store(Request $request)
    {
        // Validation des données
        $request->validate([
            'informations' => 'required|string|max:255',
            'adresse' => 'required|string|max:255',
            'tel' => 'required|string',
            'email' => 'nullable|email',
            'website' => 'nullable|string|max:255',
            'specialite' => 'nullable|string|max:255',
            'nationalite' => 'required|string|max:255',
            'description' => 'required|string',
            'photo1' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:2048',
            'photo2' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:2048',
            'photo3' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:2048',
            'validate' => 'nullable|boolean',
        ]);

        // Préparation des données
        $data = $request->except(['photo1', 'photo2', 'photo3']);

        // Par défaut, le gym créé par l'admin est validé
        $data['validate'] = $request->has('validate') ? $request->validate : true;

        // Traitement des images en BLOB
        foreach (['photo1', 'photo2', 'photo3'] as $photoField) {
            if ($request->hasFile($photoField)) {
                // Lire le contenu du fichier en binaire
                $imageData = file_get_contents($request->file($photoField)->getRealPath());
                $data[$photoField] = $imageData;
            }
        }

        // Création du gym
        Gym::create($data);

        return redirect()->route('admin.gyms.index')->with('success', 'Gym ajouté avec succès !');
    }

    public function edit($id)
    {
        $gym = Gym::findOrFail($id);
        return view('admin.register_gym.edit_gym', ['gym' => $gym, 'title' => 'Modifier le Gym']);
    }

    public function update(Request $request, $id)
    {
        // Validation des données
        $request->validate([
            'informations' => 'required|string|max:255',
            'adresse' => 'required|string|max:255',
            'tel' => 'required|string',
            'email' => 'nullable|email',
            'website' => 'nullable|string|max:255',
            'specialite' => 'nullable|string|max:255',
            'nationalite' => 'required|string|max:255',
            'description' => 'required|string',
            'photo1' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:2048',
            'photo2' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:2048',
            'photo3' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:2048',
            'validate' => 'nullable|boolean',
            'remove_photo1' => 'nullable|boolean',
            'remove_photo2' => 'nullable|boolean',
            'remove_photo3' => 'nullable|boolean',
        ]);

        try {
            // Récupérer le gym
            $gym = Gym::findOrFail($id);

            // Préparation des données
            $data = $request->except(['photo1', 'photo2', 'photo3', 'remove_photo1', 'remove_photo2', 'remove_photo3']);

            // Gestion des images
            foreach (['photo1', 'photo2', 'photo3'] as $photoField) {
                $removeField = 'remove_' . $photoField;

                // Si on veut supprimer l'image
                if ($request->has($removeField) && $request->$removeField == '1') {
                    $data[$photoField] = null;
                }
                // Si on upload une nouvelle image
                elseif ($request->hasFile($photoField)) {
                    // Lire le contenu du fichier en binaire
                    $imageData = file_get_contents($request->file($photoField)->getRealPath());
                    $data[$photoField] = $imageData;
                }
                // Sinon, on garde l'image existante
                else {
                    $data[$photoField] = $gym->$photoField;
                }
            }

            // Mise à jour du gym
            $gym->update($data);

            return redirect()->route('admin.gyms.index')->with('success', 'Gym modifié avec succès !');

        } catch (\Exception $e) {
            \Log::error('Error updating gym: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Une erreur est survenue lors de la modification.');
        }
    }

    public function updateValidate(Request $request, $id)
    {
        try {
            $gym = Gym::findOrFail($id);

            // Alterner entre validé et non validé
            $gym->validate = !$gym->validate;
            $gym->save();

            return redirect()->back()->with('success', 'Statut du gym modifié avec succès 🎉');

        } catch (\Exception $e) {
            \Log::error('Error updating gym validation: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Une erreur est survenue lors de la modification du statut.');
        }
    }

    public function destroy($id)
    {
        try {
            $gym = Gym::findOrFail($id);
            $gym->delete();

            return redirect()->back()->with('success', 'Gym supprimé avec succès 🎉');

        } catch (\Exception $e) {
            \Log::error('Error deleting gym: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Une erreur est survenue lors de la suppression.');
        }
    }
}